<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\GameDetail;
use Illuminate\Http\Request;
use App\Models\Admin;
use App\Models\User;
use App\Models\Advertisement;
use Illuminate\Support\Facades\Auth;
use File;
use ImgUploader;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;


class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:admin');
    }

    public function index()
    {
        $advertisement_counts = Advertisement::all();
        $advertisementCount = $advertisement_counts->count();

        $user_counts = User::all();
        $userCount = $user_counts->count();

        $games = GameDetail::where('game_status', 1)->get();
        $gameCounts = $games->count();

        return view('admin.dashboard', compact('advertisementCount', 'userCount','gameCounts'));
    }

    public function profile()
    {
        $admin_detail = Admin::first();
        return view('admin.profile', compact('admin_detail'));
    }


    public function updateProfile(Request $request)
    {
        $user = Auth::user(); // Retrieve the currently logged-in user

        // Validate the form fields
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email',
            'image' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'password' => 'nullable|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // Update the user's profile data
        $user->name = $request->input('name');
        $user->email = $request->input('email');

        // Check if a new password is provided and update it
        $password = $request->input('password');
        $oldPassword = $request->input('old_password');

        if (!empty($password)) {
            if (Hash::check($oldPassword, $user->password)) {
                $user->password = bcrypt($password);
            } else {
                return redirect()->back()->with('error', 'Old password is incorrect.')->withInput();
            }
        }

        // Check if a new image is provided
        if ($request->hasFile('image')) {
            // Get the uploaded image
            $image = $request->file('image');

            // Validate the image
            $validator = Validator::make(['image' => $image], [
                'image' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            // Delete the existing image (if any)
            if (!empty($user->image)) {
                $this->deleteImage($user->image);
            }

            // Upload the new image
            $imagePath = $this->uploadImage($image);

            // Update the image path in the database
            $user->image = $imagePath;
        }

        $user->save(); // Save the changes

        return redirect()->back()->with('success', 'Profile updated successfully!');
    }




    private function deleteImage($imagePath)
    {
        // Delete the image file from the storage
        if (file_exists(public_path($imagePath))) {
            unlink(public_path($imagePath));
        }
    }

    private function uploadImage($image)
    {
        // Generate a unique filename
        $filename = uniqid() . '_' . time() . '.' . $image->getClientOriginalExtension();

        // Define the storage path
        $storagePath = 'ProfilePics/';

        // Create the ProfilePics folder if it doesn't exist
        if (!File::isDirectory(public_path($storagePath))) {
            File::makeDirectory(public_path($storagePath));
        }

        // Move the uploaded image to the storage path
        $image->move(public_path($storagePath), $filename);

        // Return the image path to be stored in the database
        return $storagePath . $filename;
    }




}