<?php

namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;

use Illuminate\Http\Request;
use App\Models\GameDetail;
use App\Models\Category;
use App\Models\Ticket;
use App\Models\WinningNumber;
// use Yajra\DataTables\Facades\Datatables;
use Illuminate\Support\Facades\Validator;
// use Illuminate\Support\Facades\Datatables;
use DataTables;
use App\Models\GameRange;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\File;




class GameDetailController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth:admin');
    }

    public function index()
    {
        $gameDetails = GameDetail::all();
        return view('admin.game_details.index', compact('gameDetails'));
    }

    public function fetchData(Request $request)
    {
        if ($request->ajax()) {
            // $startDate = $request->input('start_date');
            // $query = GameDetail::query();
            // dd($data);
            
            // if ($startDate) {
                // Filter the records based on the selected start date
                // $query->whereDate('game_draw_date', $startDate);
            // }
            
            $data = GameDetail::orderByDesc('created_at')->get();

            return DataTables::of($data)
                ->addIndexColumn()
                ->editColumn('created_at', function($row) {
                    if ($row->created_at) {
                        return $row->created_at->format('Y-m-d');
                    }
                    return null;
                })
                ->addColumn('action', function ($row) {
                    $btn = '
                    <td class="text-end">
                        <a href="' . route('admin.game-details.edit', $row->id) . '" class="btn btn-outline-info btn-rounded">
                            <i class="fas fa-pen"></i>
                        </a>
                        <button type="button" class="btn btn-outline-danger btn-rounded btn-delete" data-url="' . route('admin.game-details.destroy', $row->id) . '">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                    <td><a href="'. route('admin.winning.list', $row->id) .'" class="btn btn-primary btn-sm me-1">Winning Number</a></td>
                    <td><a href="'.route('admin.manage.ticket-list', $row->id).'" class="btn btn-primary btn-sm me-1">Tickets</a></td>
                    <td><a href="'.route('admin.manage.ticket-list', $row->id).'" class="btn btn-primary btn-sm me-1">Winning Tickets</a></td>
                    <td><a href="'.route('admin.result.list', $row->id).'" class="btn btn-primary btn-sm me-1">Results</a></td>
                    ';
    
                    return $btn;
                })
                ->rawColumns(['action'])
                ->make(true);
        }
    }
    

    public function create()
    {
        // $timezones = Timezone::listIdentifiers();
        // $formattedTimezones = array_map(function ($timezone) {
        //     $moment = new Moment(null, $timezone);
        //     return [
        //         'label' => $moment->format('z'),
        //         'value' => $timezone,
        //     ];
        // }, $timezones);
        $gameCode = $this->generateGameCode();
        $categoryList = Category::latest()->get();
        return view('admin.game_details.create', compact('gameCode','categoryList'));
    }
    
    private function generateGameCode()
    {
        $gameCode = 'CODE_';
        $randomNumber = str_pad(rand(0, 9999), 4, '0', STR_PAD_LEFT);
        $gameCode .= $randomNumber;
        return $gameCode;
    }

    public function store(Request $request)
    {

        // dd($request->all());
        
        $request->validate([
            'game_code' => 'required|unique:game_details',
            'game_title' => 'required',
            'game_description' => 'required',
            'game_start_date' => 'required|date',
            'game_end_date' => 'required|date',
            'game_start_time' => 'required|date_format:H:i',
            'game_end_time' => 'required|date_format:H:i',
            'game_draw_date' => 'required|date',
            'game_draw_time' => 'required|date_format:H:i',
            'game_time_zone' => 'required',
            'game_total_allowed_ticket' => 'required|integer',
            'game_ticket_prize' => 'required|integer',
            'game_status' => 'required',
            'game_category' => 'required',
            'featured_image' => [
                    'nullable',
                    'image',
                    'mimes:jpeg,png,jpg,gif',
                    'max:5120', // Maximum file size in kilobytes (5MB)
                ],
            // 'featured_image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048', // Add image validation rules
        ]);
    
        
        $imageName = 'Game'.time() . '.' . $request->file('featured_image')->extension();
    
        if ($request->hasFile('featured_image')) {
            $image = $request->file('featured_image');
    
            // Resize the image to 150x150 pixels
            $resizedImage = Image::make($image)->fit(150, 150)->encode($image->getClientOriginalExtension());
    
            $imageName = 'Game'.time() . '.' . $image->getClientOriginalExtension();
            $resizedImage->save(public_path('game_images') . '/' . $imageName);
        }
        else {
            $imageName = null;
        }
        // $request->file('featured_image')->move(public_path('game_images'), $imageName);
    
        // $gameDetail = GameDetail::create(array_merge($request->all(), ['featured_image' => $imageName]));.
        $gameDetail = GameDetail::create(array_merge($request->except('game_min_range', 'game_max_range'), ['featured_image' => $imageName]));

        $minRanges = $request->input('game_min_range');
        $maxRanges = $request->input('game_max_range');

        // Iterate over the minRanges and maxRanges arrays and save the game ranges
        for ($i = 0; $i < count($minRanges); $i++) {
            $minRange = $minRanges[$i][0];
            $maxRange = $maxRanges[$i][0];
            $position = $i + 1;

            GameRange::create([
                'game_id' => $gameDetail->id,
                'position' => $position,
                'min_range' => $minRange,
                'max_range' => $maxRange,
            ]);
        }

    
        return redirect()->route('admin.game-details.index')->with('success', 'Game created successfully');
    }
    
    

    public function edit($id)
    {
        $gameDetail = GameDetail::findOrFail($id);
        $categoryList = Category::latest()->get();
        return view('admin.game_details.edit', compact('gameDetail','categoryList'));
    }

    public function update(Request $request, $id)
    {
        // dd($request->all());
        $gameDetail = GameDetail::findOrFail($id);
    
        $rules = [
            'game_code' => 'required|unique:game_details,game_code,' . $gameDetail->id,
            'game_title' => 'required',
            'game_description' => 'required',
            'game_start_date' => 'required|date',
            'game_end_date' => 'required|date',
            'game_start_time' => 'required|date_format:H:i:s',
            'game_end_time' => 'required|date_format:H:i:s',
            'game_draw_date' => 'required|date',
            'game_draw_time' => 'required|date_format:H:i:s',
            'game_time_zone' => 'required',
            'game_total_allowed_ticket' => 'required|integer',
            'game_ticket_prize' => 'required|integer',
            'game_status' => 'required',
            'game_category' => 'required',
        ];
    
        $validatedData = $request->validate([
            // Your existing validation rules here
            'featured_image' => 'nullable|image|max:5000', // Maximum file size: 5MB
        ]);
    
        $validator = Validator::make($request->all(), $rules);
    
        if ($validator->fails()) {
            return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
        }
    
        // Image update logic
        if ($request->hasFile('featured_image')) {
            // Remove the old image if it exists
            if ($gameDetail->featured_image) {
                $oldImagePath = public_path('game_images/' . $gameDetail->featured_image);
                if (File::exists($oldImagePath)) {
                    File::delete($oldImagePath);
                }
            }
    
            $image = $request->file('featured_image');
            $imageName = 'Game' . time() . '.' . $image->getClientOriginalExtension();
    
            // Resize the image to 150x150 pixels
            // $resizedImage = Image::make($image)->fit(150, 150)->encode($image->getClientOriginalExtension());
            // Resize the image to 150x150 pixels using the Intervention Image package
            $resizedImage = Image::make($image)->fit(150, 150);
            $resizedImage->encode($image->getClientOriginalExtension());
    
            // Save the resized image
            $resizedImage->save(public_path('game_images/' . $imageName));
    
            $gameDetail->featured_image = $imageName;
        }
    
    
        $gameDetail->game_start_time = substr($request->game_start_time, 0, 5);
        $gameDetail->game_end_time = substr($request->game_end_time, 0, 5);
    
        $gameDetail->update($request->except('featured_image'));
    
        return redirect()->route('admin.game-details.index')->with('success', 'Game updated successfully');
    
    
    

    public function deleteMethod($id)
    {
        $gameDetail = GameDetail::findOrFail($id);
    
        // Find the related tickets and delete them
        $tickets = Ticket::where('game_id', $gameDetail->id)->get();
        foreach ($tickets as $ticket) {
            // Delete the ticket
            $ticket->delete();
        }
    
        // Find the related winning numbers and delete them
        $winningNumbers = WinningNumber::where('game_id', $gameDetail->id)->get();
        foreach ($winningNumbers as $winningNumber) {
            // Delete the winning number
            $winningNumber->delete();
        }
    
        // Delete the game detail
        $gameDetail->delete();
    
        return redirect()->route('admin.game-details.index')->with('success', 'Game deleted successfully');
    }
    
    
}
